package com.b2lmobitech.fieldcloudplus.task.service

import android.Manifest
import android.app.Activity
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.location.Location
import android.util.Log
import androidx.core.app.ActivityCompat
import com.android.volley.DefaultRetryPolicy
import com.android.volley.Response
import com.android.volley.toolbox.StringRequest
import com.android.volley.toolbox.Volley
import com.b2lmobitech.fieldcloudplus.hcm.HCMURL
import com.b2lmobitech.fieldcloudplus.others.utils.Preference
import com.google.android.gms.location.Geofence
import com.google.android.gms.location.GeofencingEvent
import com.google.android.gms.location.GeofencingRequest
import com.google.android.gms.location.LocationServices

class GeofenceBroadcastReceiver : BroadcastReceiver() {

    override fun onReceive(context: Context?, intent: Intent?) {
        if (context == null || intent == null) return
        val geofencingEvent = GeofencingEvent.fromIntent(intent)
        if (geofencingEvent?.hasError() == true) {
            // Handle the error
            return
        }

        val geofenceTransition = geofencingEvent?.geofenceTransition
        if (geofenceTransition == Geofence.GEOFENCE_TRANSITION_ENTER ||
            geofenceTransition == Geofence.GEOFENCE_TRANSITION_EXIT
        ) {

            val triggeringGeofences = geofencingEvent.triggeringGeofences
            if (triggeringGeofences != null) {
                for (geofence in triggeringGeofences) {
                    val requestId = geofence.requestId
                    if (geofenceTransition == Geofence.GEOFENCE_TRANSITION_ENTER) {
                        getCurrentLocation(context, true, requestId,intent.getStringExtra("reportId")?:"")
                    } else {
                        getCurrentLocation(context, false, requestId,intent.getStringExtra("reportId")?:"")
                    }
                    Log.d("GeoFence", requestId.plus(" ").plus(geofenceTransition.toString()))
                }
            }
        }
    }

    private fun getCurrentLocation(
        context: Context?,
        isCheckIn: Boolean = false,
        facilityId: String,
        reportId:String
    ) {
        val fusedLocationClient = LocationServices.getFusedLocationProviderClient(context!!)
        if (ActivityCompat.checkSelfPermission(
                context,
                Manifest.permission.ACCESS_FINE_LOCATION
            ) != PackageManager.PERMISSION_GRANTED && ActivityCompat.checkSelfPermission(
                context,
                Manifest.permission.ACCESS_COARSE_LOCATION
            ) != PackageManager.PERMISSION_GRANTED
        ) {
            return
        }
        fusedLocationClient.lastLocation
            .addOnSuccessListener { location: Location? ->
                if (location != null) {
                    val latitude = location.latitude
                    val longitude = location.longitude
                    // Use the current location coordinates
                    updateGeoFenceToServer(context, isCheckIn, facilityId, latitude, longitude,reportId)
                } else {
                    // Handle case where location is null
                }
            }
            .addOnFailureListener { e ->
                // Handle failure to get location
            }
    }

    companion object {

        fun updateGeoFenceToServer(
            context: Context?,
            isCheckIn: Boolean = false,
            facilityId: String,
            latitude: Double,
            longitude: Double,
            reportId: String,
        ) {
            val mRequestQueue = Volley.newRequestQueue(context)
            val params: MutableMap<String, String> = java.util.HashMap()
            params["emailId"] = Preference.getInstance(context).emailId
            params["companyname"] = Preference.getInstance(context).companyName
            params["type"] = if (isCheckIn) "checkin" else "checkout"
            params["lat"] = latitude.toString()
            params["lon"] = longitude.toString()
            params["facility_id"] = facilityId
            params["reportId"] = reportId

            val postRequest: StringRequest = object : StringRequest(
                Method.POST, HCMURL.INSERT_GEO_FENCE,
                Response.Listener<String?> { response -> // response
                    Log.d("Response Geo", response!!)
                    if (isCheckIn.not()) {
                        if (context != null) unregisterGeoFence(context)
                    }
                },
                Response.ErrorListener { error -> // error
                    Log.d("Error.Response Geo", error.toString())
                }
            ) {
                override fun getParams(): Map<String, String>? {
                    return params
                }
            }
            postRequest.setRetryPolicy(DefaultRetryPolicy(240 * 1000, 1, 1.0f))
            mRequestQueue.add(postRequest)

        }

        fun registerGeoFence(
            activity: Activity,
            latitude: Double,
            longitude: Double,
            radius: Float,
            atmId: String,
            reportId:String
        ) {
            val geofencingClient = LocationServices.getGeofencingClient(activity)
            geofencingClient.removeGeofences(getGeofencePendingIntent(activity,reportId))
            // Create the geofence
            val geofence = Geofence.Builder()
                .setRequestId(atmId)
                .setCircularRegion(latitude, longitude, radius)
                .setExpirationDuration(Geofence.NEVER_EXPIRE)
                .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_ENTER or Geofence.GEOFENCE_TRANSITION_EXIT)
                .build()

            // Create the geofencing request
            val geofencingRequest = GeofencingRequest.Builder()
                .setInitialTrigger(GeofencingRequest.INITIAL_TRIGGER_ENTER)
                .addGeofence(geofence)
                .build()

            // Register the geofence
            if (ActivityCompat.checkSelfPermission(
                    activity,
                    Manifest.permission.ACCESS_FINE_LOCATION
                ) != PackageManager.PERMISSION_GRANTED
            ) {
                return
            }
            geofencingClient.addGeofences(geofencingRequest, getGeofencePendingIntent(activity,reportId))
                .addOnSuccessListener {}
                .addOnFailureListener { e -> e.printStackTrace() }
        }

        fun unregisterGeoFence(context: Context) {
            val geofencingClient = LocationServices.getGeofencingClient(context)
            try {
                geofencingClient.removeGeofences(getGeofencePendingIntent(context))
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        private fun getGeofencePendingIntent(activity: Context,reportId:String?=null): PendingIntent {
            val intent = Intent(activity, GeofenceBroadcastReceiver::class.java)
            if(reportId != null) intent.putExtra("reportId",reportId)
            return PendingIntent.getBroadcast(
                activity,
                0,
                intent,
                PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_MUTABLE
            )
        }
    }
}