package com.b2lmobitech.fieldcloudplus.task.service

import android.Manifest
import android.app.NotificationChannel
import android.app.NotificationManager
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.location.Location
import android.os.BatteryManager
import android.os.Build
import android.os.Handler
import android.os.IBinder
import android.os.Looper
import android.util.Log
import android.widget.Toast
import androidx.core.app.ActivityCompat
import androidx.core.app.NotificationCompat
import androidx.lifecycle.LifecycleService
import com.android.volley.Response
import com.android.volley.toolbox.StringRequest
import com.android.volley.toolbox.Volley
import com.b2lmobitech.fieldcloudplus.R
import com.b2lmobitech.fieldcloudplus.URL
import com.b2lmobitech.fieldcloudplus.task.service.GeofenceBroadcastReceiver.Companion.updateGeoFenceToServer
import com.b2lmobitech.fieldcloudplus.utils.DistanceCalculator
import com.b2lmobitech.fieldcloudplus.utils.LocationUtils
import com.b2lmobitech.fieldcloudplus.utils.openAppSettings
import com.google.android.gms.location.FusedLocationProviderClient
import com.google.android.gms.location.LocationServices
import com.google.android.gms.location.Priority
import kotlinx.coroutines.Runnable

class LocationBatteryService : LifecycleService() {

    private lateinit var locationClient: FusedLocationProviderClient
    private var loginCalledOnce = false

    override fun onCreate() {
        super.onCreate()
        locationClient = LocationServices.getFusedLocationProviderClient(this)
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        super.onStartCommand(intent, flags, startId)
        startForegroundService()
        intent?.run {
            startLocationUpdates(
                defectId = getStringExtra("defectId"),
                latitude = getDoubleExtra("latitude", 0.0),
                longitude = getDoubleExtra("longitude", 0.0),
                radius = getFloatExtra("radius", 0.0f),
                atmId = getStringExtra("atmId") ?: "",
            )

        }
        return START_STICKY
    }

    private fun startForegroundService() {
        createNotificationChannel()
        val notification = NotificationCompat.Builder(this, "location_channel")
            .setContentTitle("Field Cloud")
            .setContentText("Task In Progress")
            .setOngoing(true)
            .setSmallIcon(R.drawable.ic_notifications)
            .build()
        startForeground(1, notification)
    }

    private fun createNotificationChannel() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channel = NotificationChannel(
                "location_channel",
                "LocationService",
                NotificationManager.IMPORTANCE_LOW
            )
            val manager = getSystemService(NotificationManager::class.java)
            manager.createNotificationChannel(channel)
        }
    }

    private fun startLocationUpdates(
        defectId: String?,
        latitude: Double,
        longitude: Double,
        radius: Float,
        atmId: String
    ) {
        val handler = Handler(Looper.getMainLooper())
        val runnable = object : Runnable {
            override fun run() {
                if (ActivityCompat.checkSelfPermission(
                        this@LocationBatteryService,
                        Manifest.permission.ACCESS_FINE_LOCATION
                    ) != PackageManager.PERMISSION_GRANTED && ActivityCompat.checkSelfPermission(
                        this@LocationBatteryService,
                        Manifest.permission.ACCESS_COARSE_LOCATION
                    ) != PackageManager.PERMISSION_GRANTED
                ) {
                    return
                }
                val fusedLocationClient =
                    LocationServices.getFusedLocationProviderClient(this@LocationBatteryService)
                fusedLocationClient.getCurrentLocation(Priority.PRIORITY_HIGH_ACCURACY, null)
                    .addOnSuccessListener { location ->
                        if (location != null) {
                            checkBatteryStatus(
                                location,
                                defectId,
                                latitude,
                                longitude,
                                radius,
                                atmId
                            )
                        }
                    }
                handler.postDelayed(this, (60000))
            }
        }
        handler.post(runnable)
    }

    private fun checkBatteryStatus(
        location: Location,
        defectId: String?,
        latitude: Double,
        longitude: Double,
        radius: Float,
        atmId: String
    ) {
        val batteryStatus: Intent? = IntentFilter(Intent.ACTION_BATTERY_CHANGED).let { ifilter ->
            applicationContext.registerReceiver(null, ifilter)
        }

        val level: Int = batteryStatus?.getIntExtra(BatteryManager.EXTRA_LEVEL, -1) ?: -1
        val scale: Int = batteryStatus?.getIntExtra(BatteryManager.EXTRA_SCALE, -1) ?: -1

        val batteryPct: Float = level / scale.toFloat() * 100

        println("Battery: $batteryPct")
        sendLocationBatteryDetails(
            defectId ?: "",
            lat = location.latitude.toString(),
            lon = location.longitude.toString(),
            location = "${location.latitude},${location.longitude}",
            battery = batteryPct.toString()
        )
        if (loginCalledOnce.not()) {
            val distance = DistanceCalculator.distance(
                location.latitude,
                location.longitude,
                latitude,
                longitude
            )
            if (distance <= radius) {
                updateGeoFenceToServer(
                    this,
                    true,
                    atmId,
                    location.latitude,
                    location.longitude,
                    defectId ?: ""
                )
                loginCalledOnce = true
            }
        }
    }

    private fun sendLocationBatteryDetails(
        id: String,
        lat: String,
        lon: String,
        location: String,
        battery: String
    ) {
        val requestQueue = Volley.newRequestQueue(applicationContext)

        val postRequest: StringRequest = object : StringRequest(
            Method.POST, "${URL.API}/defect/defect_add_live_location.php",
            Response.Listener { response -> // response
                Log.d("Response", response!!)
                try {

                } catch (e: Exception) {
                    e.printStackTrace()
                }
            },
            Response.ErrorListener { error -> // error
                Log.d("Error.Response", error.toString())
            }
        ) {
            override fun getParams(): Map<String, String> {
                val params: MutableMap<String, String> = HashMap()
                params["report_id"] = id
                params["lat"] = lat
                params["lon"] = lon
                params["location"] = location
                params["battery"] = battery
                return params
            }
        }
        requestQueue.add(postRequest)
    }


    override fun onDestroy() {
        super.onDestroy()
    }

    override fun onBind(intent: Intent): IBinder? {
        super.onBind(intent)
        return null // Since this is a started service, not a bound one
    }

    companion object {
        fun startLocationService(
            activity: androidx.activity.ComponentActivity,
            defectId: String,
            latitude: Double,
            longitude: Double,
            radius: Float,
            atmId: String
        ) {
            LocationUtils.requestLocationPermission(activity) {
                if (it) {
                    val serviceIntent = Intent(activity, LocationBatteryService::class.java).apply {
                        putExtra("defectId", defectId)
                        putExtra("latitude", latitude)
                        putExtra("longitude", longitude)
                        putExtra("radius", radius)
                        putExtra("atmId", atmId)
                    }
                    activity.startService(serviceIntent)
                } else {
                    Toast.makeText(activity, "Enable Location Permission", Toast.LENGTH_SHORT)
                        .show()
                    activity.openAppSettings()
                }
            }
        }

        fun stopLocationService(context: Context) {
            val serviceIntent = Intent(context, LocationBatteryService::class.java)
            context.stopService(serviceIntent)
        }
    }
}