package com.b2lmobitech.fieldcloudplus.utils

import android.Manifest
import android.app.Activity
import android.content.Intent
import android.content.IntentSender
import android.content.pm.PackageManager
import android.location.Location
import android.net.Uri
import android.os.Build
import android.provider.Settings
import androidx.activity.ComponentActivity
import androidx.activity.result.ActivityResultCallback
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.IntentSenderRequest
import androidx.activity.result.contract.ActivityResultContract
import androidx.activity.result.contract.ActivityResultContracts
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import com.google.android.gms.common.api.ResolvableApiException
import com.google.android.gms.location.LocationRequest
import com.google.android.gms.location.LocationServices
import com.google.android.gms.location.LocationSettingsRequest
import com.google.android.gms.location.Priority
import com.google.android.gms.location.SettingsClient
import java.util.UUID


object LocationUtils {

    private var requestPermissionLauncher: ActivityResultLauncher<String>? = null

    @RequiresApi(Build.VERSION_CODES.TIRAMISU)
    fun requestNotificationPermission(
        activity: ComponentActivity,
        callback: (Boolean) -> Unit
    ) {
        val permission = Manifest.permission.POST_NOTIFICATIONS

        if (ContextCompat.checkSelfPermission(
                activity,
                permission
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            // Permission already granted, call the callback with true
            callback(true)
        } else {
            // Request the permission
            requestPermissionLauncher = activity.registerActivityResultLauncher(
                ActivityResultContracts.RequestPermission()
            ) { isGranted ->
                callback(isGranted)
                requestPermissionLauncher?.unregister()
            }
            requestPermissionLauncher?.launch(permission)
        }
    }

    @RequiresApi(Build.VERSION_CODES.Q)
    fun requestBackgroundLocationPermission(
        activity: ComponentActivity,
        callback: (Boolean) -> Unit
    ) {
        val permission = Manifest.permission.ACCESS_BACKGROUND_LOCATION

        if (ContextCompat.checkSelfPermission(
                activity,
                permission
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            // Permission already granted, call the callback with true
            callback(true)
        } else {
            // Request the permission
            requestPermissionLauncher = activity.registerActivityResultLauncher(
                ActivityResultContracts.RequestPermission()
            ) { isGranted ->
                callback(isGranted)
                requestPermissionLauncher?.unregister()
            }
            requestPermissionLauncher?.launch(permission)
        }
    }

    fun requestLocationPermission(
        activity: ComponentActivity,
        callback: (Boolean) -> Unit
    ) {
        if (ContextCompat.checkSelfPermission(
                activity,
                Manifest.permission.ACCESS_FINE_LOCATION
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            // Permission already granted, call the callback with true
            callback(true)
        } else {
            // Request the permission
            requestPermissionLauncher = activity.registerActivityResultLauncher(
                ActivityResultContracts.RequestPermission()
            ) { isGranted ->
                callback(isGranted)
                requestPermissionLauncher?.unregister()
            }
            requestPermissionLauncher?.launch(Manifest.permission.ACCESS_FINE_LOCATION)
        }
    }

    fun getLocation(activity: ComponentActivity, locationCallback: (location: Location) -> Unit,onPermissionDenied:()->Unit) {
        val fusedLocationClient = LocationServices.getFusedLocationProviderClient(activity)
        requestLocationPermission(activity) { permissionGranted ->
            if (permissionGranted) {
                try {
                    fusedLocationClient.getCurrentLocation(Priority.PRIORITY_HIGH_ACCURACY, null)
                        .addOnSuccessListener { location ->
                            if (location != null) {
                                locationCallback.invoke(location)
                            } else {
                                locationRequest(activity, locationCallback)
                            }
                        }
                } catch (e: SecurityException) { e.printStackTrace() }
            } else {
                onPermissionDenied.invoke()
            }
        }
    }

    private fun getLastLocation(
        activity: ComponentActivity,
        locationCallback: (location: Location) -> Unit
    ) {
        val fusedLocationClient = LocationServices.getFusedLocationProviderClient(activity)
        try {
            fusedLocationClient.lastLocation.addOnSuccessListener { location: Location? ->
                if (location != null) {
                    locationCallback.invoke(location)
                } else {
                    try {
                        fusedLocationClient.getCurrentLocation(
                            Priority.PRIORITY_HIGH_ACCURACY,
                            null
                        )
                            .addOnSuccessListener {
                                if (it != null) {
                                    locationCallback.invoke(it)
                                } else {
                                    locationRequest(activity, locationCallback)
                                }
                            }
                    } catch (e: SecurityException) {
                        //FirebaseCrashlyticsHelper.getInstance().logFireBaseCrashlytics(object {}.javaClass.name, e.message, e) //ToDo
                    }
                }
            }
        } catch (e: SecurityException) {
            //FirebaseCrashlyticsHelper.getInstance().logFireBaseCrashlytics(object {}.javaClass.name, e.message, e) //ToDo
        }
    }

    private fun locationRequest(
        activity: ComponentActivity,
        locationCallback: (location: Location) -> Unit
    ) {
        // Create the LocationRequest using the new Builder API
        val locationRequest = LocationRequest.Builder(Priority.PRIORITY_HIGH_ACCURACY, 5000)
            .setMinUpdateIntervalMillis(5000)
            .build()

        // Create the LocationSettingsRequest using the updated Builder API
        val locationSettingsRequest = LocationSettingsRequest.Builder()
            .addLocationRequest(locationRequest)
            .build()

        val client: SettingsClient = LocationServices.getSettingsClient(activity)
        val task = client.checkLocationSettings(locationSettingsRequest)
        task.addOnSuccessListener { getLastLocation(activity, locationCallback) }
        task.addOnFailureListener { exception ->
            if (exception is ResolvableApiException) {
                // Location settings are not satisfied, but this can be fixed
                // by showing the user a dialog.
                try {
                    // Show the dialog by calling startResolutionForResult(),
                    // and check the result in onActivityResult().
                    // showLocationPermissionDeniedMessage()
                    val intentSenderRequest =
                        IntentSenderRequest.Builder(exception.resolution).build()
                    var locationResultLauncher:ActivityResultLauncher<IntentSenderRequest>? = null
                     locationResultLauncher =
                        activity.registerActivityResultLauncher(ActivityResultContracts.StartIntentSenderForResult()) { result ->
                            when (result.resultCode) {
                                Activity.RESULT_OK -> getLastLocation(activity, locationCallback)
                                Activity.RESULT_CANCELED -> {
                                    //showLocationPermissionDeniedMessage() //ToDo
                                }
                            }
                            locationResultLauncher?.unregister()
                        }
                    locationResultLauncher.launch(intentSenderRequest)
                } catch (sendEx: IntentSender.SendIntentException) {
                    // Ignore the error.
                }
            }
        }
    }
}

fun Activity.openAppSettings(){
    val intent = Intent(
        Settings.ACTION_APPLICATION_DETAILS_SETTINGS, Uri.fromParts(
            "package",
            packageName, null
        )
    )
    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
    startActivity(intent)
}

fun <I, O> ComponentActivity.registerActivityResultLauncher(
    contract: ActivityResultContract<I, O>,
    callback: ActivityResultCallback<O>
): ActivityResultLauncher<I> {
    val key = UUID.randomUUID().toString()
    return activityResultRegistry.register(key, contract, callback)
}

