package com.b2lmobitech.tundraplaynew;

import android.content.Context;
import android.content.pm.ActivityInfo;
import android.net.Uri;
import android.os.Bundle;
import android.util.Base64;
import android.util.Log;
import android.view.View;
import android.widget.ImageButton;
import android.widget.Toast;

import androidx.annotation.OptIn;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.media3.common.C;
import androidx.media3.common.MediaItem;
import androidx.media3.common.util.UnstableApi;
import androidx.media3.exoplayer.ExoPlayer;
import androidx.media3.exoplayer.drm.DefaultDrmSessionManager;
import androidx.media3.exoplayer.drm.DrmSessionManager;
import androidx.media3.exoplayer.drm.FrameworkMediaDrm;
import androidx.media3.exoplayer.drm.LocalMediaDrmCallback;
import androidx.media3.exoplayer.ima.ImaAdsLoader;
import androidx.media3.exoplayer.source.DefaultMediaSourceFactory;
import androidx.media3.exoplayer.trackselection.DefaultTrackSelector;
import androidx.media3.ui.PlayerView;

import com.google.android.exoplayer2.Player;
import com.google.android.exoplayer2.ui.TrackSelectionDialogBuilder;
import com.google.android.material.bottomsheet.BottomSheetDialog;

import java.nio.charset.StandardCharsets;

public class PlayerActivity extends AppCompatActivity {

    private PlayerView playerView;
    private ExoPlayer player;
    private ImaAdsLoader adsLoader;
    private boolean isFullscreen = false;
    private String dashUrl, cid, ckey;

    private static final String VAST_AD_TAG =
            "https://processflow.in/tundra_play/app/ads/vast.php";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_player);

        playerView = findViewById(R.id.playerView);

        // ✅ Get data from intent
        dashUrl = getIntent().getStringExtra("DASH_URL");
        cid = getIntent().getStringExtra("CID");
        ckey = getIntent().getStringExtra("CKEY");

        if (dashUrl == null || cid == null || ckey == null) {
            Toast.makeText(this, "Invalid video data", Toast.LENGTH_SHORT).show();
            finish();
            return;
        }

//        playVideo(dashUrl, cid, ckey);
        enterFullscreen();

        ImageButton fullscreenBtn = playerView.findViewById(R.id.btnFullscreen);
        if (fullscreenBtn != null) {
            fullscreenBtn.setOnClickListener(v -> {
                if (isFullscreen) exitFullscreen();
                else enterFullscreen();
            });
        }
    }

    // ======================= PLAY VIDEO =======================

    @OptIn(markerClass = UnstableApi.class)
    private void playVideo(String dashUrl, String cid, String ckey) {

        // 1️⃣ Track Selector
        DefaultTrackSelector trackSelector = new DefaultTrackSelector(this);

        // 2️⃣ Ads Loader
        adsLoader = new ImaAdsLoader.Builder(this)
//                .setAdEventListener(adEvent -> {
//
//                    switch (adEvent.getType()) {
//
//                        case LOADED:
//                            // Ad metadata available
//                            break;
//
//                        case STARTED:
//                            adStartTimeMs = System.currentTimeMillis();
//                            break;
//
//                        case COMPLETED:
//                            if (adStartTimeMs != 0) {
//                                totalAdPlayedMs +=
//                                        System.currentTimeMillis() - adStartTimeMs;
//                                adStartTimeMs = 0;
//                            }
//                            break;
//
//                        case SKIPPED:
//                            if (adStartTimeMs != 0) {
//                                totalAdPlayedMs +=
//                                        System.currentTimeMillis() - adStartTimeMs;
//                                adStartTimeMs = 0;
//                            }
//                            break;
//
//                        case ALL_ADS_COMPLETED:
//                            long seconds = totalAdPlayedMs / 1000;
//                            System.out.println("Total Ad Time Played: " + seconds + " seconds");
//                            break;
//                    }
//                })
                .setAdErrorListener(adErrorEvent -> {
                    Log.e("IMA_ERROR", adErrorEvent.toString());
                    if (player != null) player.play();
                })
                .setAdEventListener(adEvent -> {
                    switch (adEvent.getType()) {
                        case CONTENT_PAUSE_REQUESTED:
                            if (player != null) player.pause();
                            break;
                        case CONTENT_RESUME_REQUESTED:
                            if (player != null) player.play();
                            break;
                    }
                })
                .build();

        // 3️⃣ ClearKey DRM (CID + CKEY)
        byte[] kidBytes = Base64.decode(cid, Base64.DEFAULT);
        byte[] keyBytes = Base64.decode(ckey, Base64.DEFAULT);

        String kidUrl = Base64.encodeToString(
                kidBytes,
                Base64.URL_SAFE | Base64.NO_PADDING | Base64.NO_WRAP
        );

        String keyUrl = Base64.encodeToString(
                keyBytes,
                Base64.URL_SAFE | Base64.NO_PADDING | Base64.NO_WRAP
        );

        String clearKeyJson = String.format(
                "{\"keys\":[{\"kty\":\"oct\",\"k\":\"%s\",\"kid\":\"%s\"}],\"type\":\"temporary\"}",
                keyUrl, kidUrl
        );

        byte[] drmData = clearKeyJson.getBytes(StandardCharsets.UTF_8);

        DrmSessionManager drmSessionManager =
                new DefaultDrmSessionManager.Builder()
                        .setUuidAndExoMediaDrmProvider(
                                C.CLEARKEY_UUID,
                                FrameworkMediaDrm.DEFAULT_PROVIDER
                        )
                        .build(new LocalMediaDrmCallback(drmData));

        // 4️⃣ MediaSource Factory (DRM + ADS)
        DefaultMediaSourceFactory mediaSourceFactory =
                new DefaultMediaSourceFactory(this)
                        .setDrmSessionManagerProvider(unused -> drmSessionManager)
                        .setAdsLoaderProvider(unused -> adsLoader)
                        .setAdViewProvider(playerView);

        // 5️⃣ Player
        player = new ExoPlayer.Builder(this)
                .setTrackSelector(trackSelector)
                .setMediaSourceFactory(mediaSourceFactory)
                .build();

        playerView.setPlayer(player);
        adsLoader.setPlayer(player);
        playerView.setUseController(true);

        // 6️⃣ MediaItem
        MediaItem mediaItem = new MediaItem.Builder()
                .setUri(dashUrl)
                .setDrmConfiguration(
                        new MediaItem.DrmConfiguration.Builder(C.CLEARKEY_UUID).build()
                )
                .setAdsConfiguration(
                        new MediaItem.AdsConfiguration.Builder(
                                Uri.parse(VAST_AD_TAG)
                        ).build()
                )
                .build();

        player.setMediaItem(mediaItem);
        player.prepare();
//        player.play();
        player.setPlayWhenReady(true);

    }

    // ======================= SETTINGS =======================

    private void showSpeedDialog() {
        if (player == null) return;
        String[] speeds = {"0.5x", "1x", "1.25x", "1.5x", "2x"};
        float[] values = {0.5f, 1f, 1.25f, 1.5f, 2f};

        new AlertDialog.Builder(this)
                .setTitle("Playback Speed")
                .setItems(speeds, (dialog, i) -> player.setPlaybackSpeed(values[i]))
                .show();
    }

    private void showQualityDialog() {
        if (player == null) return;
        new TrackSelectionDialogBuilder(
                (Context) this,
                "Select Quality",
                (Player) player,
                C.TRACK_TYPE_VIDEO
        ).build().show();
    }

    private void showAudioDialog() {
        if (player == null) return;
        new TrackSelectionDialogBuilder(
                (Context) this,
                "Audio Track",
                (Player) player,
                C.TRACK_TYPE_AUDIO
        ).build().show();
    }

    private void showSettingsSheet() {
        BottomSheetDialog dialog = new BottomSheetDialog(this);
        View view = getLayoutInflater().inflate(R.layout.dialog_player_settings, null);
        dialog.setContentView(view);

        view.findViewById(R.id.option_speed).setOnClickListener(v -> {
            dialog.dismiss();
            showSpeedDialog();
        });

        view.findViewById(R.id.option_audio).setOnClickListener(v -> {
            dialog.dismiss();
            showAudioDialog();
        });

        view.findViewById(R.id.option_quality).setOnClickListener(v -> {
            dialog.dismiss();
            showQualityDialog();
        });

        dialog.show();
    }

    // ======================= FULLSCREEN =======================

    private void enterFullscreen() {
        isFullscreen = true;
        setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE);
        getWindow().getDecorView().setSystemUiVisibility(
                View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY
                        | View.SYSTEM_UI_FLAG_FULLSCREEN
                        | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
        );
    }

    private void exitFullscreen() {
        isFullscreen = false;
        setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);
        getWindow().getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_VISIBLE);
    }

    @Override
    public void onBackPressed() {
        if (isFullscreen) exitFullscreen();
        else super.onBackPressed();
    }

    // ======================= LIFECYCLE =======================

    @Override
    protected void onStop() {
        super.onStop();

        if (player != null) {
            playerView.setPlayer(null);
            player.release();
            player = null;
        }
    }


    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (adsLoader != null) {
            adsLoader.release();
            adsLoader = null;
        }
    }

    @Override
    protected void onStart() {
        super.onStart();
        if (player == null) {
            playVideo(dashUrl, cid, ckey);
        }
    }

}
